# 機能設計書 26-Pages Router

## 概要

本ドキュメントは、Next.jsにおけるPages Router機能の設計を記述する。Pages Routerは、`pages/`ディレクトリによるファイルシステムベースの従来型ルーティング機構であり、Next.jsの初期から存在するルーティングの基盤である。

### 本機能の処理概要

**業務上の目的・背景**：Webアプリケーションのルーティングを、ファイルシステムの構造に基づいて自動的に構成する仕組みを提供する。Pages Routerは`pages/`ディレクトリ配下のファイルを自動的にルートとして認識し、SSR/SSG/CSRといった複数のレンダリング戦略をサポートする。App Router登場以前からのNext.jsの主要ルーティング機構であり、多くの既存プロジェクトで使用されている。

**機能の利用シーン**：全てのNext.js Webアプリケーションのページルーティング（Pages Routerベースのプロジェクト）。ページコンポーネントの定義、データフェッチ関数（getServerSideProps/getStaticProps/getStaticPaths）との統合、カスタムApp/Document/Errorページの提供など。

**主要な処理内容**：
1. `pages/`ディレクトリのファイルスキャンとルート生成
2. ページコンポーネントのレンダリング（SSR/SSG/CSR）
3. データフェッチ関数の実行とpropsの注入
4. カスタムApp（`_app.tsx`）によるグローバルラッパー提供
5. カスタムDocument（`_document.tsx`）によるHTML構造制御
6. クライアントサイドルーティングとプリフェッチ

**関連システム・外部連携**：ビルドシステム（Webpack/Turbopack）、サーバーランタイム（Node.js/Edge）、データフェッチAPI（getServerSideProps等）、next/routerフックと連携する。

**権限による制御**：フレームワークレベルでの組み込み認証はないが、`getServerSideProps`内でリダイレクトベースの認証制御が可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | エラーページ (_error) | 補助機能 | Pages Routerのルーティング機構を通じてエラーページを表示 |
| 2 | ドキュメント (_document) | 補助機能 | Pages Routerのページファイル管理を通じたCSS・スクリプトの読み込み制御 |
| 3 | アプリケーションラッパー (_app) | 補助機能 | Pages Routerを通じたページコンポーネントの受け渡しとレンダリング |

## 機能種別

ルーティング制御 / レンダリング統合

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| req | IncomingMessage | Yes | Node.jsのHTTPリクエスト | 有効なHTTPリクエスト |
| res | ServerResponse | Yes | Node.jsのHTTPレスポンス | 有効なHTTPレスポンス |
| page | string | Yes | リクエストされたページパス | pages/以下の有効なパス |
| query | NextParsedUrlQuery | Yes | パースされたURLクエリ | オブジェクト形式 |
| renderOpts | RenderOpts | Yes | レンダリングオプション | 必要なプロパティを含む |

### 入力データソース

- HTTPリクエスト（クライアントからのナビゲーション/直接アクセス）
- `pages/`ディレクトリのファイル構造

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| RenderResult | RenderResult | レンダリング結果（HTMLストリームまたはJSON） |
| headers | Headers | レスポンスヘッダー |
| revalidate | number \| false | ISR再検証間隔 |
| pageData | object | クライアントサイドナビゲーション用ページデータ |

### 出力先

- HTTPレスポンス（HTML/JSON形式）

## 処理フロー

### 処理シーケンス

```
1. ルーティング解決
   └─ URLパスからpages/ディレクトリ内のページファイルを特定
2. ページモジュールのロード
   └─ PagesRouteModuleによるページコンポーネントとデータフェッチ関数の読み込み
3. データフェッチ実行
   └─ getServerSideProps/getStaticPropsの実行によるprops取得
4. レンダリング
   └─ renderToHTMLImplによるReactコンポーネントのHTML生成
5. レスポンス送信
   └─ HTML/JSONをクライアントに送信
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト] --> B[ルーティング解決]
    B --> C[PagesRouteModule.render]
    C --> D{データフェッチ関数?}
    D -->|getServerSideProps| E[SSRデータフェッチ]
    D -->|getStaticProps| F[SSGデータフェッチ]
    D -->|getInitialProps| G[初期データフェッチ]
    D -->|なし| H[自動静的最適化]
    E --> I[renderToHTMLImpl]
    F --> I
    G --> I
    H --> I
    I --> J[App + Document + Componentでレンダリング]
    J --> K[RenderResult返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-26-01 | 排他的データフェッチ | getServerSidePropsとgetStaticPropsは同一ページで共存不可 | 常時 |
| BR-26-02 | getInitialProps排他 | getInitialPropsとgetServerSideProps/getStaticPropsは共存不可 | 常時 |
| BR-26-03 | 自動静的最適化 | getInitialProps/getServerSideProps/getStaticPropsが未定義で、AppもデフォルトのgetInitialPropsを使用する場合は自動的に静的ページとして最適化 | isAutoExport判定時 |
| BR-26-04 | 動的SSGパス必須 | 動的ルートでSSG（getStaticProps）を使用する場合はgetStaticPathsが必須 | isSSG && pageIsDynamic |
| BR-26-05 | ステータスページ制約 | `/404`や`/500`ページではgetInitialProps/getServerSidePropsは使用不可 | STATIC_STATUS_PAGES該当時 |

### 計算ロジック

自動静的最適化判定：`isAutoExport = !hasPageGetInitialProps && defaultAppGetInitialProps && !isSSG && !getServerSideProps`

## データベース操作仕様

### 操作別データベース影響一覧

該当なし。Pages Router自体はデータベースに直接アクセスしない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | SSG_GET_INITIAL_PROPS_CONFLICT | getInitialPropsとSSGが共存 | いずれかを削除 |
| - | SERVER_PROPS_GET_INIT_PROPS_CONFLICT | getInitialPropsとgetServerSidePropsが共存 | いずれかを削除 |
| - | SERVER_PROPS_SSG_CONFLICT | getServerSidePropsとSSGが共存 | いずれかを削除 |
| - | 無効なコンポーネント | defaultエクスポートがReactコンポーネントでない | 有効なReactコンポーネントをエクスポート |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- 自動静的最適化によりデータフェッチ不要なページはビルド時にHTMLを生成
- クライアントサイドナビゲーション時はJSON形式のページデータをプリフェッチ
- `getStaticProps`の`revalidate`オプションによるISR対応

## セキュリティ考慮事項

- `getServerSideProps`/`getStaticProps`のコードはクライアントバンドルに含まれない
- プレビューモード（Draft Mode）のCookieベースのアクセス制御

## 備考

- Pages RouterはApp Routerと並行して使用可能（段階的な移行をサポート）
- 新規プロジェクトではApp Routerの使用が推奨される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

PagesRouteModuleの型定義とモジュール構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | module.ts | `packages/next/src/server/route-modules/pages/module.ts` | `PagesUserlandModule`型（38-63行目）でページモジュールの構造を理解 |

**読解のコツ**: `PagesUserlandModule`はdefaultエクスポート（コンポーネント）、config、そして3つのデータフェッチ関数（getStaticProps/getStaticPaths/getServerSideProps）のオプショナルプロパティを持つ。

#### Step 2: エントリーポイントを理解する

PagesRouteModule.render()がメインエントリーポイント。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | module.ts | `packages/next/src/server/route-modules/pages/module.ts` | `render()`メソッド（136-154行目）がrenderToHTMLImplに委譲 |

**主要処理フロー**:
1. **141行目**: `renderToHTMLImpl()`への委譲
2. **147-149行目**: `App`と`Document`コンポーネントの注入

#### Step 3: レンダリングロジックを理解する

`render.tsx`の巨大なrenderToHTMLImpl関数。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | render.tsx | `packages/next/src/server/render.tsx` | `renderToHTMLImpl()`関数のデータフェッチ → レンダリング → レスポンス生成の全体フロー |

**主要処理フロー**:
- **519行目**: isSSG判定（getStaticPropsの有無）
- **547-551行目**: isAutoExport判定
- **565-601行目**: データフェッチ関数の排他制約チェック

### プログラム呼び出し階層図

```
PagesRouteModule.render() [module.ts:136]
    |
    +-- renderToHTMLImpl() [render.tsx]
            |
            +-- isDynamicRoute() - 動的ルート判定
            +-- isSSG / isAutoExport 判定
            +-- データフェッチ関数の排他チェック
            +-- getServerSideProps() / getStaticProps() 実行
            +-- Reactコンポーネントレンダリング
            |       +-- App コンポーネント
            |       +-- Document コンポーネント
            |       +-- Page コンポーネント
            +-- RenderResult 生成
```

### データフロー図

```
[入力]                     [処理]                           [出力]

HTTPリクエスト        ──> ルーティング解決               ──> ページモジュール特定
                          |
ページモジュール      ──> データフェッチ関数実行          ──> pageProps
                          |
App + Document +     ──> renderToHTMLImpl()               ──> RenderResult (HTML)
Page + pageProps          |
                          |
RenderResult         ──> レスポンス送信                  ──> HTTPレスポンス
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| module.ts | `packages/next/src/server/route-modules/pages/module.ts` | ソース | PagesRouteModuleのメイン実装 |
| render.tsx | `packages/next/src/server/render.tsx` | ソース | Pages Routerのレンダリングロジック |
| pages-handler.ts | `packages/next/src/server/route-modules/pages/pages-handler.ts` | ソース | Pagesリクエストハンドラ |
| module.render.ts | `packages/next/src/server/route-modules/pages/module.render.ts` | ソース | レンダリングモジュール |
| pages-route-definition.ts | `packages/next/src/server/route-definitions/pages-route-definition.ts` | ソース | ルート定義型 |
| router.ts | `packages/next/src/shared/lib/router/router.ts` | ソース | クライアントサイドルーター |
| next-server.ts | `packages/next/src/server/next-server.ts` | ソース | プロダクションサーバー |
